/* ...............................................................

	WindowColors
	Copyright 1997-8 Steve Klingsporn <moofie@pobox.com>
	Based on WindowShade by Marco Nelissen <marcone@xs4all.nl>
	
		File:	ColorLabel.cpp
	
	Contains:	Implementation of "color divot label" BView
				subclass.
	
	   Notes:	None.
	   
   ............................................................... */

#ifndef _COLOR_LABEL_H
#include "ColorLabel.h"
#endif

/* ...............................................................	
	ColorLabel Constructor
	Sets the view color to transparent (we draw all our bits).
   ............................................................... */

ColorLabel::ColorLabel(BRect frame, const char *name, 
								 window_color which)
		    :BView(frame, name, B_FOLLOW_NONE, B_WILL_DRAW | B_NAVIGABLE),
		     _labelStringWidth(-1),
		     _which(which)
{
	SetViewColor(RGB_COLOR_216);
}


/* ...............................................................	
	ColorLabel::Draw()
	Draws the divot label.
   ............................................................... */

void ColorLabel::Draw(BRect updateRect)
{
	/* Draw the label, if need be, with the right hiliting */
	if (updateRect.Intersects(LabelBounds()))
	{
		SetLowColor(RGB_COLOR_216);
		
		/* Keep point sizes in check (Sorry, Marco) */
		BFont labelFont = BFont(be_plain_font);
		if (labelFont.Size() > 14)
			labelFont.SetSize(14);
					
		/* Calculate string width if not cached */
		if (_labelStringWidth == -1)
			_labelStringWidth = labelFont.StringWidth(Name());
					
		BRect bounds = LabelBounds();
		FillRect(bounds & updateRect, B_SOLID_LOW);
		
		float y = (3 * (bounds.Height() / 4));
		
		MovePenTo(bounds.left + 8, y);
		SetHighColor(BLACK_COLOR);
		SetFont(&labelFont);
		DrawString(Name());
		
		/* Draw the focus underline if we're focused and
		   our window is active */
		if (IsFocus() && Window()->IsActive())
		{
			y += 3;
			BeginLineArray(2);
			AddLine(BPoint(bounds.left + 8, y), 
					BPoint(bounds.left + 8 + _labelStringWidth, y), 
					keyboard_navigation_color());
			AddLine(BPoint(bounds.left + 8, y + 1), 
					BPoint(bounds.left + 8 + 
					_labelStringWidth, y + 1), 
					WHITE_COLOR);
			EndLineArray();
		}
	}
	
	/* Draw the divot frame, if need be */
	if (updateRect.Intersects(DivotBounds()))
	{
		BRect bounds = DivotBounds();
		
		/* Determine if we should be drawn focused */
		bool drawFocused = (IsFocus() && Window()->IsActive());
		
		BeginLineArray(8);
		AddLine(BPoint(bounds.left, bounds.top), 
				BPoint(bounds.right - 1, bounds.top), 
				RGB_COLOR_184);
		AddLine(BPoint(bounds.left, bounds.top + 1), 
				BPoint(bounds.left, 
				bounds.bottom - 1), RGB_COLOR_184);
		AddLine(BPoint(bounds.right, bounds.top), 
				BPoint(bounds.right, bounds.bottom), 
				WHITE_COLOR);
		AddLine(BPoint(bounds.right - 1, bounds.bottom), 
				BPoint(bounds.left, 
				bounds.bottom), WHITE_COLOR);
		
		/* Change the color if we're focused */
		rgb_color color = RGB_COLOR_160;
		if (drawFocused)
			color = keyboard_navigation_color();
			
		AddLine(BPoint(bounds.left + 1, bounds.top + 1), 
				BPoint(bounds.right - 2, 
				bounds.top + 1), color);
		AddLine(BPoint(bounds.left + 1, bounds.top + 2), 
				BPoint(bounds.left + 1, 
				bounds.bottom - 2), color);
				
		/* Don't change the color unless we're unfocused */
		if (! drawFocused)
			color = RGB_COLOR_216;
			
		AddLine(BPoint(bounds.right - 1, bounds.top + 1), 
				BPoint(bounds.right - 1, 
				bounds.bottom - 1), color);
		AddLine(BPoint(bounds.right - 2, bounds.bottom - 1), 
				BPoint(bounds.left + 1, 
				bounds.bottom - 1), color);
		EndLineArray();
				
		/* Fill in the color divot */
		bounds.InsetBy(2, 2);
		SetHighColor(Color());
		FillRect(bounds & updateRect);		
	}
}


/* ...............................................................	
	ColorLabel::MessageReceived()
	Called when a BMessage is received by this view.  If the
	message contains RGB color data in the accepted format from
	such messages, we assume the color embedded in the message.
   ............................................................... */

void ColorLabel::MessageReceived(BMessage *message)
{
	rgb_color *color;
	ssize_t	  size;
	
	if (message->FindData("RGBColor", B_RGB_COLOR_TYPE,
						  &color, &size) == B_NO_ERROR)
		SetColor(*color);
	else
		BView::MessageReceived(message);
}


/* ...............................................................
	ColorLabel::WindowActivated()
	Overridden to redraw the view if we're focused when the window
	gains and loses activation.  There's no sense in displaying
	the focus when our window isn't active.
   ............................................................... */

void ColorLabel::WindowActivated(bool state)
{
#pragma unused(state)
	if (IsFocus())
		Draw(Bounds());
}


/* ...............................................................	
	ColorLabel::MouseDown()
	Overridden to allow the user to right-drag a color message
	in the accepted format from the divot, and to set this
	view to the focused view.
   ............................................................... */

void ColorLabel::MouseDown(BPoint where)
{
	uint32 buttons;
	GetMouse(&where, &buttons);
	
	if (buttons & B_SECONDARY_MOUSE_BUTTON)
	{
		/* Create and drag a color message */
		BMessage message = BMessage(B_SIMPLE_DATA);
		rgb_color color = Color();
		message.AddData("RGBColor", B_RGB_COLOR_TYPE, &color,
						 sizeof(rgb_color));
		BBitmap *bitmap = new BBitmap(BRect(0, 0, 10, 10), 
									  B_COLOR_8_BIT);
		uint8 index = index_for_color(Color());
		BScreen *screen = new BScreen();
		color_space depth = screen->ColorSpace();
		delete screen;
		
		switch (depth)
		{
			case B_COLOR_8_BIT:
			default: /* fix for 1.0 to right color bitmap */
				::memset(bitmap->Bits(), index, 
						 bitmap->BitsLength());
				break;
		}
		DragMessage(&message, bitmap, BPoint(5, 5));
	}
	else
	{
		/* Set this view to the focused view */
		if (! IsFocus())
			MakeFocus(true);
	}
}


/* ...............................................................	
	ColorLabel::Color()
	Returns the color for this label.
   ............................................................... */

rgb_color ColorLabel::Color()
{
	return ParentWindowView()->ParentPanel()->Color(_which);
}


/* ...............................................................	
	ColorLabel::SetColor()
	Sets the color for this label.  Sets the color in the model
	(stored in the WindowColorsPanel parent view) properly.
   ............................................................... */

void ColorLabel::SetColor(rgb_color color, bool quietly)
{
	WindowColorsPanel *panel = ParentWindowView()->ParentPanel();
	
	if (! compare_colors(color, panel->Color(_which)))
	{
		ParentWindowView()->ParentPanel()->SetColor(_which, color);
		if (! IsHidden())
		{
			/* Redraw our view to update the color, and update
			   the color in the ColorPicker */
			Draw(DivotBounds());
			if (! quietly)
				ParentWindowView()->ParentPanel()->
					GetColorPicker()->SetValueQuietly(color);
		}
	}
}


/* ...............................................................	
	ColorLabel::DivotBounds
	Returns the bounds of the divot part of the view (used for
	simple redraw testing).
   ............................................................... */

BRect ColorLabel::DivotBounds()
{
	BRect bounds = Bounds();
	bounds.right = bounds.left + 33;
	return bounds;
}


/* ...............................................................	
	ColorLabel::LabelBounds()
	Returns the bounds of the label part of the view (used for
	simple redraw testing).
   ............................................................... */

BRect ColorLabel::LabelBounds()
{
	BRect bounds = Bounds();
	bounds.left += 34;
	return bounds;
}


/* ...............................................................
	ColorLabel::ParentWindowView()
	Returns the parent view with the right type.
   ............................................................... */

WindowView *ColorLabel::ParentWindowView()
{
	return ((WindowView *)Parent());
}


/* ...............................................................
	ColorLabel::MakeFocus()
	Overridden to correctly handle the focus.  A ColorLabel
	will only become focus when selected or tabbed to without
	the Option key down.
   ............................................................... */

void ColorLabel::MakeFocus(bool focused)
{
	if (focused != IsFocus())
	{
		BView::MakeFocus(focused);
		WindowView *parent = ParentWindowView();
		WindowColorsPanel *panel = parent->ParentPanel();
				
		/* If we should activate the parent WindowView, do so */
		if (panel->ActiveWindowView() != parent)
			if (! (modifiers() & B_OPTION_KEY))
				panel->SetActiveWindowView(parent);

		/* Update the color control */
		ColorPicker *colorControl = panel->GetColorPicker();
		if (colorControl != NULL)
			colorControl->SetValueQuietly(Color());
		
		/* Redraw the entire view */			
		Draw(Bounds());
	}
}


/* ...............................................................	
	ColorLabel::Which()
	Returns the window_color this view represents.
   ............................................................... */

window_color ColorLabel::Which()
{
	return _which;
}


/* ...............................................................	
	ColorLabel::KeyDown()
	Redirect key events to the color control, but strip out the
	return key, because it sets the color control to black, and
	that's yucky.
   ............................................................... */

void ColorLabel::KeyDown(const char *bytes, int32 numBytes)
{
	if (numBytes == 1 && bytes[0] != B_RETURN)
	{
		ParentWindowView()->ParentPanel()->GetColorPicker()->
			KeyDown(bytes, numBytes);
	}
}